#!/usr/bin/env python3
import argparse
import datetime as dt
import os
import subprocess
import sys
from dataclasses import dataclass
from typing import Iterable, Optional

try:
    from PIL import Image, ImageDraw, ImageFont
except ImportError:
    print("This script requires Pillow. Install with: pip install pillow", file=sys.stderr)
    sys.exit(1)


# ---------- data structures ----------

@dataclass
class EventInfo:
    date: dt.date
    label: str
    week_index: int


@dataclass
class GoalInfo:
    start_date: dt.date
    end_date: dt.date
    label: str
    week_indices: set[int]


# ---------- argument parsing ----------

def parse_args():
    parser = argparse.ArgumentParser(description="Generate a memento mori calendar image.")
    parser.add_argument(
        "--birthdate",
        required=True,
        help="Birthdate in YYYY-MM-DD format (e.g. 1990-01-01).",
    )
    parser.add_argument(
        "--years",
        type=int,
        default=80,
        help="Number of years/rows to show (default: 80).",
    )
    parser.add_argument(
        "--output",
        default="~/Pictures/memento_mori.png",
        help="Output image path (default: ~/Pictures/memento_mori.png).",
    )
    parser.add_argument(
        "--last-state-file",
        default="~/.memento_mori_last_weeks",
        help="File storing the last weeks-lived count (default: ~/.memento_mori_last_weeks).",
    )
    parser.add_argument(
        "--box-size",
        type=int,
        default=10,
        help="Size of each week box in pixels (default: 10).",
    )
    parser.add_argument(
        "--spacing",
        type=int,
        default=2,
        help="Spacing between boxes in pixels (default: 2).",
    )
    parser.add_argument(
        "--margin",
        type=int,
        default=20,
        help="Minimum outer margin (in pixels) around the grid (default: 20).",
    )
    parser.add_argument(
        "--filled-color",
        default="#000000",
        help="Color for weeks that have passed (default: black).",
    )
    parser.add_argument(
        "--empty-color",
        default="#CCCCCC",
        help="Color for weeks that have not yet passed (default: light gray).",
    )
    parser.add_argument(
        "--background-color",
        default="#FFFFFF",
        help="Background color (default: white).",
    )
    parser.add_argument(
        "--set-wallpaper",
        default=None,
        help=(
            "Shell command to set wallpaper. Use '{image}' where the image path should go. "
            "Example (nitrogen): \"nitrogen --set-zoom-fill {image} --save\""
        ),
    )
    parser.add_argument(
        "--image-size",
        default=None,
        help=(
            "Target image size in WIDTHxHEIGHT, e.g. 1920x1080. "
            "The grid is centered inside this size using padding."
        ),
    )
    parser.add_argument(
        "--aspect-ratio",
        default=None,
        help=(
            "Target aspect ratio W:H, e.g. 16:9. "
            "The image is expanded (by increasing padding) to match this ratio, "
            "without changing box size."
        ),
    )
    parser.add_argument(
        "--force",
        action="store_true",
        help="Force generation without caring about last update."
    )

    parser.add_argument(
        "--years-per-group",
        type=int,
        default=5,
        help="Insert an extra vertical gap every N years (default: 5). Use <=0 to disable grouping.",
    )
    parser.add_argument(
        "--vertical-gap-size",
        type=int,
        default=-1,
        help="Extra vertical gap height in pixels between year groups (default: same as box size).",
    )

    parser.add_argument(
        "--last-week-color",
        default="#FF0000",
        help="Color for the most recent (last active) week (default: red).",
    )
    parser.add_argument(
        "--text-color",
        default="#000000",
        help="Color for text labels (default: black).",
    )

    parser.add_argument(
        "--horizontal-parts",
        type=int,
        default=2,
        help=(
            "Divide each year row into this many horizontal parts, inserting gaps between them "
            "(default: 2). Use 1 to disable extra horizontal gaps."
        ),
    )
    parser.add_argument(
        "--horizontal-gap-size",
        type=int,
        default=-1,
        help="Extra horizontal gap width in pixels between parts (default: same as box size).",
    )

    parser.add_argument(
        "--font-size-multiplier",
        type=float,
        default=1.2,
        help=(
            "Font size as a multiple of box size (default: 1.2). "
            "Actual font size in pixels = box_size * font_size_multiplier."
        ),
    )
    parser.add_argument(
        "--font-size-multiplier-stats",
        type=float,
        default=None,
        help=(
            "Font size multiplier for the stats block on the left. "
            "If not set, falls back to --font-size-multiplier."
        ),
    )
    parser.add_argument(
        "--font-size-multiplier-notes",
        type=float,
        default=None,
        help=(
            "Font size multiplier for the events/goals notes on the right. "
            "If not set, falls back to --font-size-multiplier."
        ),
    )
    parser.add_argument(
        "--font-path",
        default=None,
        help=(
            "Optional path to a TTF/OTF font file for labels. "
            "If not provided, the script tries a serif font if available, "
            "and falls back to Pillow's default font."
        ),
    )

    parser.add_argument(
        "--theme",
        choices=["warm", "cool", "mono", "warm-dark", "cool-dark", "mono-dark"],
        default=None,
        help="Apply a built-in color theme (overrides the color-related options).",
    )


    parser.add_argument(
        "--show-stats",
        action="store_true",
        help="Show a stats block with today, precise age, weeks lived/left, and this-year stats.",
    )


    parser.add_argument(
        "--highlight-current-year",
        action="store_true",
        help="Highlight the current year row with a background color.",
    )
    parser.add_argument(
        "--current-year-background-color",
        default="#FFF4CC",
        help="Background color for the current year row when highlighted.",
    )

    parser.add_argument(
        "--show-start-end-labels",
        action="store_true",
        help="Show labels at the beginning and end of life (e.g. 'Start' and 'Fin.').",
    )
    parser.add_argument(
        "--start-label",
        default="Start",
        help="Text label for the beginning of the life grid (default: 'Start').",
    )
    parser.add_argument(
        "--end-label",
        default="Fin.",
        help="Text label for the end of the life grid (default: 'Fin.').",
    )

    parser.add_argument(
        "--expected-years",
        type=int,
        default=None,
        help=(
            "Expected lifespan in years. If set and less than --years, draws a horizontal marker "
            "line at that age and optionally shades later years as 'bonus time'."
        ),
    )
    parser.add_argument(
        "--expectation-line-color",
        default="#000000",
        help="Color for the life-expectancy marker line.",
    )
    parser.add_argument(
        "--shade-bonus-years",
        action="store_true",
        help="Shade years beyond --expected-years using --bonus-background-color.",
    )
    parser.add_argument(
        "--bonus-background-color",
        default="#F5F5F5",
        help="Background color for years beyond expected lifespan when shading is enabled.",
    )

    parser.add_argument(
        "--event",
        action="append",
        default=[],
        help="Life event in the form YYYY-MM-DD[:Description].",
    )
    parser.add_argument(
        "--event-file",
        action="append",
        default=[],
        help=(
            "Path to a file with events, one per line: YYYY-MM-DD[:Description]. "
            "Empty lines and lines starting with '#' are ignored."
        ),
    )
    parser.add_argument(
        "--event-color",
        default="#0000FF",
        help="Color for event markers (box outline).",
    )

    parser.add_argument(
        "--goal",
        action="append",
        default=[],
        help="Goal period in the form START_DATE:END_DATE[:Description].",
    )
    parser.add_argument(
        "--goal-file",
        action="append",
        default=[],
        help=(
            "Path to a file with goals, one per line: START_DATE:END_DATE[:Description]. "
            "Empty lines and lines starting with '#' are ignored."
        ),
    )
    parser.add_argument(
        "--goal-color",
        default="#008000",
        help="Color for goal band markers (box outline).",
    )

    return parser.parse_args()


# ---------- helpers ----------

def parse_image_size(size_str: str):
    try:
        parts = size_str.lower().split("x")
        if len(parts) != 2:
            raise ValueError
        w = int(parts[0].strip())
        h = int(parts[1].strip())
        if w <= 0 or h <= 0:
            raise ValueError
        return w, h
    except Exception:
        raise ValueError("Invalid --image-size, expected format WIDTHxHEIGHT, e.g. 1920x1080")


def parse_aspect_ratio(ratio_str: str) -> float:
    try:
        parts = ratio_str.split(":")
        if len(parts) != 2:
            raise ValueError
        w = float(parts[0].strip())
        h = float(parts[1].strip())
        if w <= 0 or h <= 0:
            raise ValueError
        return w / h
    except Exception:
        raise ValueError("Invalid --aspect-ratio, expected format W:H, e.g. 16:9")


def weeks_lived_since(birthdate: dt.date, today: dt.date) -> int:
    """
    Return a week index compatible with a 52-weeks-per-row life grid:
    - First, count full calendar years already lived (whole birthdays passed).
    - Then add whole weeks since the last birthday.
    - Each year contributes at most 52 weeks, so rows stay aligned.
    """
    if today <= birthdate:
        return 0

    years = compute_integer_age_years(birthdate, today)

    year = birthdate.year + years
    try:
        last_birthday = birthdate.replace(year=year)
    except ValueError:
        if birthdate.month == 2 and birthdate.day == 29:
            last_birthday = dt.date(year, 3, 1)
        else:
            raise

    days_since_birthday = (today - last_birthday).days
    weeks_since_birthday = max(0, days_since_birthday // 7)

    if weeks_since_birthday > 51:
        weeks_since_birthday = 51

    return years * 52 + weeks_since_birthday


def read_last_weeks(path: str):
    if not os.path.exists(path):
        return None
    try:
        with open(path, "r", encoding="utf-8") as f:
            return int(f.read().strip())
    except Exception:
        return None


def write_last_weeks(path: str, weeks: int) -> None:
    try:
        with open(path, "w", encoding="utf-8") as f:
            f.write(str(weeks))
    except Exception as e:
        print(f"Warning: could not write last weeks file '{path}': {e}", file=sys.stderr)


def build_font(box_size: int, font_size_multiplier: float, font_path: Optional[str]) -> ImageFont.ImageFont:
    size = max(1, int(round(box_size * font_size_multiplier)))

    if font_path:
        try:
            return ImageFont.truetype(font_path, size=size)
        except OSError as e:
            print(f"Warning: could not load font from '{font_path}': {e}. Falling back to defaults.", file=sys.stderr)

    candidate_paths = [
        "/usr/share/fonts/cantarell/Cantarell-VF.otf",
        "/usr/share/fonts/truetype/dejavu/DejaVuSerif.ttf",
        "/usr/share/fonts/truetype/dejavu/DejaVuSans.ttf",
        "/Library/Fonts/Times New Roman.ttf",
        "Times New Roman.ttf",
    ]
    for candidate in candidate_paths:
        try:
            return ImageFont.truetype(candidate, size=size)
        except OSError:
            continue

    return ImageFont.load_default()


def apply_theme(args) -> None:
    if args.theme is None:
        return

    if args.theme == "warm":
        args.background_color = "#fdf6e3"
        args.filled_color = "#f39c4d"
        args.empty_color = "#dcd3c9"
        args.last_week_color = "#e74c3c"
        args.text_color = "#5b4636"
        args.current_year_background_color = "#fdebd0"
        args.bonus_background_color = "#f9f2e7"
        args.event_color = "#8e44ad"
        args.goal_color = "#27ae60"
    elif args.theme == "cool":
        args.background_color = "#f5f7fb"
        args.filled_color = "#4a90e2"
        args.empty_color = "#d0d7e6"
        args.last_week_color = "#e74c3c"
        args.text_color = "#2c3e50"
        args.current_year_background_color = "#dde9fb"
        args.bonus_background_color = "#edf1fb"
        args.event_color = "#9b59b6"
        args.goal_color = "#16a085"
    elif args.theme == "mono":
        args.background_color = "#FFFFFF"
        args.filled_color = "#000000"
        args.empty_color = "#E0E0E0"
        args.last_week_color = "#FF0000"
        args.text_color = "#000000"
        args.current_year_background_color = "#F5F5F5"
        args.bonus_background_color = "#F0F0F0"
        args.event_color = "#0000FF"
        args.goal_color = "#008000"
    elif args.theme == "warm-dark":
        args.background_color = "#1b130c"
        args.filled_color = "#ffb066"
        args.empty_color = "#4a3b2e"
        args.last_week_color = "#ff4d4d"
        args.text_color = "#f5e6d5"
        args.current_year_background_color = "#2a2016"
        args.bonus_background_color = "#241b12"
        args.event_color = "#c39bd3"
        args.goal_color = "#58d68d"
    elif args.theme == "cool-dark":
        args.background_color = "#0d1117"
        args.filled_color = "#61a8ff"
        args.empty_color = "#30363d"
        args.last_week_color = "#ff6b6b"
        args.text_color = "#e6edf3"
        args.current_year_background_color = "#161b22"
        args.bonus_background_color = "#11161d"
        args.event_color = "#a66bff"
        args.goal_color = "#2ecc71"
    elif args.theme == "mono-dark":
        args.background_color = "#000000"
        args.filled_color = "#f5f5f5"
        args.empty_color = "#3b3b3b"
        args.last_week_color = "#ff5555"
        args.text_color = "#f5f5f5"
        args.current_year_background_color = "#141414"
        args.bonus_background_color = "#101010"
        args.event_color = "#5dade2"
        args.goal_color = "#58d68d"


def compute_integer_age_years(birthdate: dt.date, today: dt.date) -> int:
    years = today.year - birthdate.year
    if (today.month, today.day) < (birthdate.month, birthdate.day):
        years -= 1
    return max(0, years)


def read_specs_from_files(paths: Iterable[str]) -> list[str]:
    specs: list[str] = []
    for path in paths:
        if not path:
            continue
        real = os.path.expanduser(path)
        if not os.path.exists(real):
            print(f"Warning: event/goal file '{real}' does not exist; skipping.", file=sys.stderr)
            continue
        try:
            with open(real, "r", encoding="utf-8") as f:
                for line in f:
                    s = line.strip()
                    if not s or s.startswith("#"):
                        continue
                    specs.append(s)
        except Exception as e:
            print(f"Warning: could not read file '{real}': {e}", file=sys.stderr)
    return specs


def process_width(label: str):
    width = 80
    second_width = 60
    padding = 52
    first_chunk = label[:width]
    remaining = label[width:]
    return [
        first_chunk
    ] + [
        " " * padding + remaining[i : i + second_width]
        for i in range(0, len(remaining), second_width)
    ]


def parse_event_specs(
    specs: Iterable[str],
    birthdate: dt.date,
    total_weeks: int,
) -> tuple[list[EventInfo], set[int]]:
    events: list[EventInfo] = []
    weeks: set[int] = set()
    for spec in specs:
        s = spec.strip()
        if not s:
            continue
        parts = s.split(":", 1)
        date_str = parts[0].strip()
        label = parts[1].strip() if len(parts) > 1 else ""
        try:
            date_val = dt.date.fromisoformat(date_str)
        except ValueError:
            print(
                f"Warning: could not parse event date '{spec}', expected YYYY-MM-DD[:Description].",
                file=sys.stderr,
            )
            continue

        if date_val < birthdate:
            continue

        week_idx = weeks_lived_since(birthdate, date_val)
        if 0 <= week_idx < total_weeks:
            events.append(EventInfo(date=date_val, label=label, week_index=week_idx))
            weeks.add(week_idx)
    return events, weeks


def parse_goal_specs(
    specs: Iterable[str],
    birthdate: dt.date,
    total_weeks: int,
) -> tuple[list[GoalInfo], set[int]]:
    goals: list[GoalInfo] = []
    all_weeks: set[int] = set()
    for spec in specs:
        s = spec.strip()
        if not s:
            continue
        parts = s.split(":", 2)
        if len(parts) < 2:
            print(f"Warning: could not parse goal '{spec}', expected START:END[:Description].", file=sys.stderr)
            continue
        start_str = parts[0].strip()
        end_str = parts[1].strip()
        label = parts[2].strip() if len(parts) > 2 else ""
        try:
            start_date = dt.date.fromisoformat(start_str)
            end_date = dt.date.fromisoformat(end_str)
        except ValueError:
            print(f"Warning: invalid goal dates in '{spec}', expected YYYY-MM-DD.", file=sys.stderr)
            continue

        if end_date < start_date:
            start_date, end_date = end_date, start_date

        if end_date < birthdate:
            continue

        clamped_start = max(start_date, birthdate)

        start_week = weeks_lived_since(birthdate, clamped_start)
        end_week = weeks_lived_since(birthdate, end_date)

        if start_week >= total_weeks:
            continue

        end_week = min(end_week, total_weeks - 1)
        if end_week < start_week:
            continue

        weeks = set(range(start_week, end_week + 1))
        if not weeks:
            continue
        goals.append(GoalInfo(start_date=start_date, end_date=end_date, label=label, week_indices=weeks))
        all_weeks.update(weeks)
    return goals, all_weeks


# ---------- image generation ----------

def generate_image(
    output_path: str,
    weeks_lived: int,
    years: int,
    box_size: int,
    spacing: int,
    margin: int,
    filled_color: str,
    empty_color: str,
    background_color: str,
    image_size=None,
    aspect_ratio: Optional[float] = None,
    years_per_group: Optional[int] = 5,
    vertical_gap_size: int = 10,
    last_week_color: Optional[str] = None,
    text_color: str = "#000000",
    horizontal_parts: int = 2,
    horizontal_gap_size: int = 10,
    font_size_multiplier: float = 1.2,
    font_path: Optional[str] = None,
    font_size_multiplier_stats: Optional[float] = None,
    font_size_multiplier_notes: Optional[float] = None,
    show_stats: bool = False,
    stats_text_lines: Optional[list[str]] = None,
    show_start_end_labels: bool = False,
    start_label: str = "Start",
    end_label: str = "Fin.",
    highlight_current_year: bool = False,
    current_year_index: Optional[int] = None,
    current_year_background_color: str = "#FFF4CC",
    expected_years: Optional[int] = None,
    expectation_line_color: str = "#000000",
    shade_bonus_years: bool = False,
    bonus_background_color: str = "#F5F5F5",
    event_weeks: Optional[set[int]] = None,
    event_color: str = "#0000FF",
    goal_weeks: Optional[set[int]] = None,
    goal_color: str = "#008000",
    event_infos: Optional[list[EventInfo]] = None,
    goal_infos: Optional[list[GoalInfo]] = None,
):
    total_weeks = years * 52
    event_weeks = event_weeks or set()
    goal_weeks = goal_weeks or set()
    stats_text_lines = stats_text_lines or []
    event_infos = event_infos or []
    goal_infos = goal_infos or []

    font_main = build_font(box_size, font_size_multiplier, font_path)

    if font_size_multiplier_stats is not None:
        font_stats = build_font(box_size, font_size_multiplier_stats, font_path)
    else:
        font_stats = font_main

    if font_size_multiplier_notes is not None:
        font_notes = build_font(box_size, font_size_multiplier_notes, font_path)
    else:
        font_notes = font_main


    if years_per_group is not None and years_per_group > 0:
        group_size = years_per_group
        block_spacing = vertical_gap_size
        num_gaps_v = (years - 1) // group_size
    else:
        group_size = None
        block_spacing = 0
        num_gaps_v = 0

    if horizontal_parts is not None and horizontal_parts > 1:
        segments = horizontal_parts
        raw_positions: list[int] = []
        for g in range(1, segments):
            pos = round(g * 52 / segments) - 1
            pos = max(0, min(52 - 2, pos))
            raw_positions.append(pos)
        gap_positions = sorted(set(raw_positions))
        num_gaps_h = len(gap_positions)
    else:
        segments = None
        gap_positions = []
        num_gaps_h = 0

    grid_width_core = (
        52 * box_size
        + (52 - 1) * spacing
        + num_gaps_h * horizontal_gap_size
    )
    grid_height = (
        years * box_size
        + (years - 1) * spacing
        + num_gaps_v * block_spacing
    )

    label_area_width = 0
    label_padding = 0
    labels_for_width: list[str] = []
    if group_size is not None and years > 0:
        labels_for_width.append(str(years))
    if show_start_end_labels:
        labels_for_width.append(start_label)
        labels_for_width.append(end_label)

    if labels_for_width:
        tmp_img = Image.new("RGB", (1, 1))
        tmp_draw = ImageDraw.Draw(tmp_img)
        max_label_w = 0
        for lbl in labels_for_width:
            bbox = tmp_draw.textbbox((0, 0), lbl, font=font_main)
            w = bbox[2] - bbox[0]
            if w > max_label_w:
                max_label_w = w
        label_padding = max(spacing * 2, box_size // 2)
        label_area_width = label_padding + max_label_w

    notes_area_width = 0
    notes_padding = 0
    notes_lines: list[str] = []
    notes_text_width = 0
    notes_text_height = 0

    if event_infos:
        notes_lines.append("Events:")
        for ev in sorted(event_infos, key=lambda e: e.date):
            if ev.label:
                notes_lines += process_width(f"[{ev.date.isoformat()}] {ev.label}")
            else:
                notes_lines.append(f"[{ev.date.isoformat()}]")
    if goal_infos:
        if notes_lines:
            notes_lines.append("")
        notes_lines.append("Goals:")
        for g in sorted(goal_infos, key=lambda gi: gi.start_date):
            period = f"{g.start_date.isoformat()}–{g.end_date.isoformat()}"
            if g.label:
                notes_lines += process_width(f"[{period}] {g.label}")
            else:
                notes_lines.append(f"[{period}]")

    notes_text = "\n".join(notes_lines)
    if notes_lines:
        notes_padding = max(spacing * 2, box_size // 2)
        tmp_img2 = Image.new("RGB", (1, 1))
        tmp_draw2 = ImageDraw.Draw(tmp_img2)
        bbox_notes = tmp_draw2.multiline_textbbox((0, 0), notes_text, font=font_notes, spacing=2)
        notes_text_width = bbox_notes[2] - bbox_notes[0]
        notes_text_height = bbox_notes[3] - bbox_notes[1]

    grid_width_all = grid_width_core + label_area_width

    margin_x = margin
    margin_y = margin
    base_width = grid_width_all + 2 * margin_x
    base_height = grid_height + 2 * margin_y

    if image_size is not None:
        target_w, target_h = image_size
        if grid_width_all > target_w or grid_height > target_h:
            raise ValueError(
                f"Grid ({grid_width_all}x{grid_height}) does not fit into requested image size "
                f"{target_w}x{target_h}. Increase box size/spacing or image size."
            )
        width = target_w
        height = target_h
        margin_x = (width - grid_width_all) // 2
        margin_y = (height - grid_height) // 2
    else:
        width = base_width
        height = base_height
        if aspect_ratio is not None:
            current_ratio = width / height
            target_ratio = aspect_ratio
            if abs(current_ratio - target_ratio) > 1e-6:
                if current_ratio < target_ratio:
                    width = int(round(height * target_ratio))
                else:
                    height = int(round(width / target_ratio))
                margin_x = max(margin, (width - grid_width_all) // 2)
                margin_y = max(margin, (height - grid_height) // 2)

    image = Image.new("RGB", (width, height), background_color)
    draw = ImageDraw.Draw(image)

    def row_y0(row: int) -> int:
        if group_size is not None and group_size > 0:
            gaps_above = row // group_size
            return margin_y + row * (box_size + spacing) + gaps_above * block_spacing
        return margin_y + row * (box_size + spacing)

    def col_x0(col: int) -> int:
        if not gap_positions:
            return margin_x + col * (box_size + spacing)
        gaps_before = sum(1 for gp in gap_positions if gp < col)
        return margin_x + col * (box_size + spacing) + gaps_before * horizontal_gap_size

    if (
        expected_years is not None
        and shade_bonus_years
        and 0 <= expected_years < years
    ):
        for row in range(expected_years, years):
            y0_row = row_y0(row)
            y1_row = y0_row + box_size
            draw.rectangle(
                [margin_x, y0_row, margin_x + grid_width_core, y1_row],
                fill=bonus_background_color,
            )

    if (
        highlight_current_year
        and current_year_index is not None
        and 0 <= current_year_index < years
    ):
        y0_row = row_y0(current_year_index)
        y1_row = y0_row + box_size
        draw.rectangle(
            [margin_x, y0_row, margin_x + grid_width_core, y1_row],
            fill=current_year_background_color,
        )

    if weeks_lived > 0:
        last_week_index_in_grid = min(weeks_lived - 1, total_weeks - 1)
    else:
        last_week_index_in_grid = None

    for row in range(years):
        y0_row = row_y0(row)
        for col in range(52):
            index = row * 52 + col
            if index >= total_weeks:
                break

            x0 = col_x0(col)
            y0 = y0_row
            x1 = x0 + box_size
            y1 = y0 + box_size

            if last_week_index_in_grid is not None and index == last_week_index_in_grid and last_week_color:
                color = last_week_color
            elif index < weeks_lived:
                color = filled_color
            else:
                color = empty_color

            draw.rectangle([x0, y0, x1, y1], fill=color)

            if index in goal_weeks:
                draw.rectangle([x0, y0, x1, y1], outline=goal_color, width=2)

            if index in event_weeks:
                draw.rectangle([x0 + 1, y0 + 1, x1 - 1, y1 - 1], outline=event_color, width=1)

    if expected_years is not None and 0 < expected_years < years:
        line_y = row_y0(expected_years) - spacing // 2
        draw.line(
            [(margin_x, line_y), (margin_x + grid_width_core, line_y)],
            fill=expectation_line_color,
            width=1,
        )

    if group_size is not None and group_size > 0 and num_gaps_v > 0:
        x_grid_right = margin_x + grid_width_core
        x_base = x_grid_right + (label_padding if label_area_width > 0 else max(spacing * 2, box_size // 2))
        for g in range(1, num_gaps_v + 1):
            age = g * group_size
            label = str(age)

            last_row_before_gap = g * group_size - 1
            next_row_after_gap = last_row_before_gap + 1

            y0_last = row_y0(last_row_before_gap)
            y_bottom_last = y0_last + box_size
            y0_next = row_y0(next_row_after_gap)
            y_top_next = y0_next

            y_top_gap = y_bottom_last + spacing
            y_bottom_gap = y_top_next - spacing
            y_center = (y_top_gap + y_bottom_gap) // 2

            bbox = draw.textbbox((0, 0), label, font=font_main)
            text_h = bbox[3] - bbox[1]

            x_text = x_base
            y_text = int(y_center - text_h / 2)
            draw.text((x_text, y_text), label, fill=text_color, font=font_main)

    if show_start_end_labels:
        x_grid_right = margin_x + grid_width_core
        x_base = x_grid_right + (label_padding if label_area_width > 0 else max(spacing * 2, box_size // 2))

        if years > 0:
            y_center_start = row_y0(0) + box_size // 2
            bbox_s = draw.textbbox((0, 0), start_label, font=font_main)
            text_h_s = bbox_s[3] - bbox_s[1]
            y_text_s = int(y_center_start - text_h_s / 2)
            draw.text((x_base, y_text_s), start_label, fill=text_color, font=font_main)

            y_center_end = row_y0(years - 1) + box_size // 2
            bbox_e = draw.textbbox((0, 0), end_label, font=font_main)
            text_h_e = bbox_e[3] - bbox_e[1]
            y_text_e = int(y_center_end - text_h_e / 2)
            draw.text((x_base, y_text_e), end_label, fill=text_color, font=font_main)

    if show_stats and stats_text_lines:
        stats_text = "\n".join(stats_text_lines)
        bbox = draw.multiline_textbbox((0, 0), stats_text, font=font_stats, spacing=2)
        stats_w = bbox[2] - bbox[0]
        stats_h = bbox[3] - bbox[1]

        side_left_x0 = 0
        side_left_x1 = margin_x
        side_left_width = max(0, side_left_x1 - side_left_x0)

        stats_padding = spacing * 2
        vertical_region_top = margin_y + stats_padding
        vertical_region_bottom = height - margin_y - stats_padding
        vertical_region_height = max(0, vertical_region_bottom - vertical_region_top)

        if side_left_width >= stats_w and vertical_region_height >= stats_h:
            x_stats = side_left_x0 + (side_left_width - stats_w) // 2
            y_stats = vertical_region_top + (vertical_region_height - stats_h) // 2
        else:
            x_stats = (width - stats_w) // 2
            y_stats = stats_padding

        draw.multiline_text((x_stats, y_stats), stats_text, fill=text_color, font=font_stats, spacing=2)

    if notes_lines:
        side_right_x0 = margin_x + grid_width_core + label_area_width
        side_right_x1 = width
        side_right_width = max(0, side_right_x1 - side_right_x0)

        vertical_region_top = margin_y + notes_padding
        vertical_region_bottom = height - margin_y - notes_padding
        vertical_region_height = max(0, vertical_region_bottom - vertical_region_top)

        if side_right_width >= notes_text_width and vertical_region_height >= notes_text_height:
            x_notes = side_right_x0 + (side_right_width - notes_text_width) // 2
            y_notes = vertical_region_top + (vertical_region_height - notes_text_height) // 2
        else:
            x_notes = side_right_x0 + max(0, side_right_width - notes_text_width)
            y_notes = vertical_region_top

        draw.multiline_text((x_notes, y_notes), notes_text, fill=text_color, font=font_notes, spacing=2)

    os.makedirs(os.path.dirname(output_path), exist_ok=True)
    image.save(output_path)


# ---------- wallpaper helper ----------

def set_wallpaper(cmd_template: str, image_path: str):
    cmd = cmd_template.format(image=image_path)
    try:
        subprocess.run(cmd, shell=True, check=False)
    except Exception as e:
        print(f"Warning: failed to set wallpaper with command '{cmd}': {e}", file=sys.stderr)


# ---------- main ----------

def main():
    args = parse_args()

    try:
        birthdate = dt.date.fromisoformat(args.birthdate)
    except ValueError:
        print("Error: --birthdate must be in YYYY-MM-DD format.", file=sys.stderr)
        sys.exit(1)

    if args.image_size and args.aspect_ratio:
        print("Error: Use either --image-size or --aspect-ratio, not both.", file=sys.stderr)
        sys.exit(1)

    apply_theme(args)

    image_size = None
    if args.image_size:
        try:
            image_size = parse_image_size(args.image_size)
        except ValueError as e:
            print(f"Error: {e}", file=sys.stderr)
            sys.exit(1)

    aspect_ratio: Optional[float] = None
    if args.aspect_ratio:
        try:
            aspect_ratio = parse_aspect_ratio(args.aspect_ratio)
        except ValueError as e:
            print(f"Error: {e}", file=sys.stderr)
            sys.exit(1)

    today = dt.date.today()
    weeks_now = weeks_lived_since(birthdate, today)
    
    last_state_file = os.path.expanduser(args.last_state_file)
    last_weeks = read_last_weeks(last_state_file)
    
    if last_weeks is not None and last_weeks == weeks_now and not args.force:
        return
    
    output_path = os.path.abspath(os.path.expanduser(args.output))
    
    years_per_group = args.years_per_group if args.years_per_group > 0 else None
    vertical_gap_size = args.vertical_gap_size if args.vertical_gap_size > 0 else args.box_size
    horizontal_parts = args.horizontal_parts if args.horizontal_parts > 1 else 1
    horizontal_gap_size = args.horizontal_gap_size if args.horizontal_gap_size > 0 else args.box_size
    
    stats_lines: list[str] = []
    
    current_year_index = compute_integer_age_years(birthdate, today)
    total_weeks = args.years * 52
    weeks_left_total = max(0, total_weeks - weeks_now)
    
    days_lived = (today - birthdate).days
    age_years_float = max(0.0, days_lived / 365.2425)
    
    weeks_this_life_year = 0
    weeks_in_life_year = 52
    weeks_left_this_life_year = 0
    
    if current_year_index < args.years:
        life_year_start_index = current_year_index * 52
        weeks_this_life_year = max(0, weeks_now - life_year_start_index)
        if weeks_this_life_year > weeks_in_life_year:
            weeks_this_life_year = weeks_in_life_year
        weeks_left_this_life_year = max(0, weeks_in_life_year - weeks_this_life_year)
    
    if args.show_stats:
        stats_lines.append(f"Today: {today.isoformat()} · Age {age_years_float:.2f} years")
        stats_lines.append(f"Weeks lived: {weeks_now} · Weeks left: {weeks_left_total}")
        stats_lines.append(
            f"This life year (age {current_year_index}→{current_year_index + 1}): "
            f"{weeks_this_life_year} / {weeks_in_life_year} "
            f"(left: {weeks_left_this_life_year})"
        )

    event_specs = list(args.event)
    event_specs.extend(read_specs_from_files(args.event_file))
    goal_specs = list(args.goal)
    goal_specs.extend(read_specs_from_files(args.goal_file))

    event_infos, event_weeks = parse_event_specs(event_specs, birthdate, total_weeks)
    goal_infos, goal_weeks = parse_goal_specs(goal_specs, birthdate, total_weeks)

    expected_years = args.expected_years

    try:
        generate_image(
            output_path=output_path,
            weeks_lived=weeks_now,
            years=args.years,
            box_size=args.box_size,
            spacing=args.spacing,
            margin=args.margin,
            filled_color=args.filled_color,
            empty_color=args.empty_color,
            background_color=args.background_color,
            image_size=image_size,
            aspect_ratio=aspect_ratio,
            years_per_group=years_per_group,
            vertical_gap_size=vertical_gap_size,
            last_week_color=args.last_week_color,
            text_color=args.text_color,
            horizontal_parts=horizontal_parts,
            horizontal_gap_size=horizontal_gap_size,
            font_size_multiplier=args.font_size_multiplier,
            font_path=args.font_path,
            font_size_multiplier_stats=args.font_size_multiplier_stats,
            font_size_multiplier_notes=args.font_size_multiplier_notes,
            show_stats=args.show_stats,
            stats_text_lines=stats_lines,
            show_start_end_labels=args.show_start_end_labels,
            start_label=args.start_label,
            end_label=args.end_label,
            highlight_current_year=args.highlight_current_year,
            current_year_index=current_year_index,
            current_year_background_color=args.current_year_background_color,
            expected_years=expected_years,
            expectation_line_color=args.expectation_line_color,
            shade_bonus_years=args.shade_bonus_years,
            bonus_background_color=args.bonus_background_color,
            event_weeks=event_weeks,
            event_color=args.event_color,
            goal_weeks=goal_weeks,
            goal_color=args.goal_color,
            event_infos=event_infos,
            goal_infos=goal_infos,
        )
    except ValueError as e:
        print(f"Error while generating image: {e}", file=sys.stderr)
        sys.exit(1)

    write_last_weeks(last_state_file, weeks_now)

    if args.set_wallpaper:
        set_wallpaper(args.set_wallpaper, output_path)


if __name__ == "__main__":
    main()

