# Memento Mori Wallpaper Generator

This script generates a “life calendar” image where each box represents one week of your life. It can:

* Highlight the most recent week and, optionally, the current year row.
* Mark individual life events and multi-week goal or project periods.
* Show a statistics panel with your current age, weeks lived/left, and per-year progress.
* Draw a life expectancy marker and optionally shade years beyond that as “bonus time”.
* Optionally call a wallpaper-setting command (for example `nitrogen`) to apply the image.

It is designed to be run automatically (via `systemd`, cron, or file-change triggers) so that your wallpaper stays up to date.

The life grid is always 52 weeks per row and `--years` rows high.

---

## Features

* Weekly life grid with 52 columns (weeks) per row and configurable number of rows (`--years`).
* Caching via a “last weeks lived” file so the image is only regenerated when needed.
* Layout control: box size, spacing, margins, grouping of years, horizontal breaks in each year, image size, and aspect ratio.
* Built-in color themes (light/dark variants).
* Life events and goals:

  * Events: single weeks with optional labels.
  * Goals: multi-week bands with labels; also listed in a text panel.
* Optional left-hand stats block summarizing:

  * Today’s date.
  * Age in years (as a float).
  * Weeks lived and remaining within the configured grid.
  * Progress in the current “life year” (weeks used and left).
* Optional life-expectancy line and “bonus years” shading.
* Optional wallpaper integration via a custom shell command.

---

## Requirements

* Python 3.8 or newer.
* [Pillow](https://python-pillow.org/) for image generation:

  ```bash
  pip install pillow
  ```

Optional, for automatic wallpaper setting:

* A wallpaper tool that accepts an image path, such as `nitrogen`.

On Arch Linux, a typical setup might look like:

```bash
sudo pacman -S python python-pip nitrogen
pip install --user pillow
```

---

## Installation

Assume you place the script at:

```text
~/.config/memento-mori/memento-mori.py
```

Make it executable:

```bash
chmod +x ~/.config/memento-mori/memento-mori.py
```

Ensure `~/.config/memento-mori` is on your `PATH`, or adjust the paths in the examples.

---

## Basic Usage

The only required argument is your birthdate:

```bash
python ~/.config/memento-mori/memento-mori.py \
  --birthdate 2000-01-01
```

Notes:

* `--birthdate` must be in `YYYY-MM-DD` format.
* By default, the script:

  * Generates `~/Pictures/memento_mori.png`.
  * Uses 80 years (rows) and 52 weeks per row.
  * Uses simple black/white/gray colors.
  * Does not modify your wallpaper unless you pass `--set-wallpaper`.

A more customized 16:9 wallpaper for a 1920×1080 display, including stats, theming, events, and goals:

```bash
python ~/.config/memento-mori/memento-mori.py \
  --birthdate 2000-01-01 \
  --image-size 1920x1080 \
  --spacing 2 \
  --margin 40 \
  --box-size 9 \
  --theme warm-dark \
  --show-stats \
  --highlight-current-year \
  --show-start-end-labels \
  --expected-years 60 \
  --expectation-line-color "#FFFFFF" \
  --font-size-multiplier 1.6 \
  --goal-file ~/goals.txt \
  --event-file ~/events.txt \
  --font-path /usr/share/fonts/cantarell/Cantarell-VF.otf \
  --output ~/Pictures/memento_mori.png \
  --force
```

To set the wallpaper using `nitrogen`:

```bash
python ~/.config/memento-mori/memento-mori.py \
  --birthdate 2000-01-01 \
  ... \
  --set-wallpaper "nitrogen --set-zoom-fill {image} --save"
```

The `{image}` placeholder in `--set-wallpaper` is replaced by the absolute output path.

---

## Event and Goal Files

Events and goals can be provided either:

* Inline on the command line with `--event` and `--goal`, or
* From one or more files via `--event-file` and `--goal-file`.

Paths are expanded with `~` (`os.path.expanduser`), and missing files are skipped with a warning.

### Event files

Each non-empty, non-comment line has the form:

```text
YYYY-MM-DD[:Description]
```

Example:

```text
# events.txt
2005-09-01:Started school
2018-06-15:Graduated high school
2022-10-01:Moved to new city
```

Details:

* Lines starting with `#` or that are empty are ignored.
* The date must be `YYYY-MM-DD`.
* The text after the first `:` (if present) is used as the label shown in the notes panel.
* Dates earlier than your birthdate are ignored.
* Events whose computed week index falls outside the configured grid (`--years`) are ignored.

You can pass multiple event files:

```bash
--event-file ~/events.txt \
--event-file ~/.config/memento/events_extra.txt
```

You can also specify events directly:

```bash
--event 2010-01-01:New year resolution \
--event 2018-06-15:Graduated high school
```

### Goal / project files

Each non-empty, non-comment line has the form:

```text
START_DATE:END_DATE[:Description]
```

Example:

```text
# goals.txt
2024-01-01:2024-06-30:Learn Rust
2024-07-01:2024-12-31:Marathon training
2025-01-01:2025-12-31:Deep work on research project
```

Details:

* `START_DATE` and `END_DATE` must be `YYYY-MM-DD`.
* If `END_DATE < START_DATE`, the script swaps them automatically.
* Goals that end before your birthdate are ignored.
* Goals that partially overlap your life or the configured grid are clamped:

  * Start is clamped to `max(START_DATE, birthdate)`.
  * Weeks beyond the last visible week (`--years * 52`) are dropped.
* The weeks in the goal range are:

  * Outlined on the grid (default green, configurable via `--goal-color`).
  * Listed in the notes panel as `[START–END] Description`, where START and END are the original dates.

Multiple goal files are allowed:

```bash
--goal-file ~/goals.txt \
--goal-file ~/.config/memento/work_goals.txt
```

Or specify them inline:

```bash
--goal 2024-01-01:2024-06-30:Learn Rust \
--goal 2024-07-01:2024-12-31:Marathon training
```

---

## How Weeks and Years Are Computed

The script uses a 52-weeks-per-row model:

* Each row corresponds to one “life year”.
* Each year contributes at most 52 weeks, regardless of leap years.

Internally:

* Your integer age in years is computed from `birthdate` and today’s date.
* For each completed birthday:

  * The script adds 52 weeks to your week index.
* For the current year (since your last birthday):

  * It counts whole weeks (integer division by 7 of the days since the last birthday).
  * The result is capped at 51 so that each row has 0–51 week indices.

Special cases:

* If today is on or before your birthdate, `weeks_lived` is 0.
* If your birthday is February 29, then in non-leap years the “last birthday” is treated as March 1 for week counting.
* If your age exceeds `--years`, weeks beyond the grid are not drawn. `weeks_lived` can be larger than `--years * 52`, but the grid is limited to `--years * 52` boxes.

The stats block (when enabled with `--show-stats`) reports:

* `Age` as a floating-point number in years (days lived divided by 365.2425).
* `Weeks lived` as the current week index.
* `Weeks left` as `max(0, years * 52 - weeks_lived)`.
* For the current life year, weeks used and weeks left, but only within the configured grid; if you are older than `--years`, the “this life year” numbers will effectively be based on the last visible year.

---

## Behavior and Caching

The script caches the last computed `weeks_lived` value in a file (by default `~/.memento_mori_last_weeks`).

On each run:

1. It computes `weeks_lived` for today.
2. It reads `last_weeks` from the cache file if it exists and can be parsed as an integer.
3. If `last_weeks == weeks_lived` and `--force` is not set:

   * The script exits without generating an image.
   * The wallpaper command (if provided) is not run.

This makes it safe to call the script frequently (e.g. every 30 minutes) without unnecessary work. Use `--force` for changes that are not tied to week progression, such as:

* Adjusting layout, colors, or fonts.
* Modifying event/goal files.
* Changing the output image path or wallpaper command.

The cache file path is expanded with `~`. If the file is missing or unreadable, the script behaves as if no previous value exists and regenerates the image.

---

## Command-line Options

This section summarizes the main arguments. Defaults are in parentheses.

### Core grid and cache

* `--birthdate YYYY-MM-DD`
  Required. Your date of birth.

* `--years N` (`80`)
  Number of years (rows) to display. The grid will contain `N * 52` weeks.

* `--output PATH` (`~/Pictures/memento_mori.png`)
  Output image path. The directory part is used with `os.makedirs`.
  Important: the path should include a directory component (for example `~/Pictures/...`), not just a bare filename.

* `--last-state-file PATH` (`~/.memento_mori_last_weeks`)
  Path to the cache file storing the last “weeks lived” value.

* `--force`
  Ignore the cache and regenerate unconditionally.

### Layout and sizing

* `--box-size PX` (`10`)
  Size of each week box in pixels.

* `--spacing PX` (`2`)
  Spacing between boxes in pixels.

* `--margin PX` (`20`)
  Minimum outer margin (in pixels) around the grid before any aspect-ratio padding.

* `--image-size WxH`
  Target image size in pixels (for example `1920x1080`).
  The grid is centered inside this canvas; margins are adjusted accordingly.
  If the grid does not fit inside the requested size, the script exits with a `ValueError` explaining the required minimum size.

* `--aspect-ratio W:H`
  Target aspect ratio (for example `16:9`).
  The script starts from the natural size of the grid (plus margins) and then expands width or height (by increasing padding) to achieve the requested ratio.
  You must not use `--image-size` and `--aspect-ratio` together; the script enforces this.

### Colors and themes

* `--filled-color HEX` (`#000000`)
  Fill color for weeks that have already passed.

* `--empty-color HEX` (`#CCCCCC`)
  Fill color for weeks that have not yet occurred.

* `--background-color HEX` (`#FFFFFF`)
  Image background color.

* `--last-week-color HEX` (`#FF0000`)
  Fill color for the most recent (last active) week cell.

* `--text-color HEX` (`#000000`)
  Color for labels and text (stats, year labels, notes).

* `--event-color HEX` (`#0000FF`)
  Outline color for event markers.

* `--goal-color HEX` (`#008000`)
  Outline color for goal bands.

* `--theme NAME`
  Apply a built-in color theme, overriding the color-related options above. One of:
  `warm`, `cool`, `mono`, `warm-dark`, `cool-dark`, `mono-dark`.

  With the current implementation, when `--theme` is set, any explicit color options (`--filled-color`, `--empty-color`, etc.) are overwritten by the theme.

### Grouping and gaps

* `--years-per-group N` (`5`)
  Insert an extra vertical gap every N years. Use `<= 0` to disable grouping.

* `--vertical-gap-size PX` (`box-size` by default)
  Extra vertical gap between year groups. If set to a non-positive value, the script uses `box_size`.

* `--horizontal-parts N` (`2`)
  Split each year row into N segments with slightly larger gaps between segments. Use `1` to disable segment gaps. The script automatically computes gap positions to divide 52 weeks into roughly equal parts.

* `--horizontal-gap-size PX` (`box-size` by default)
  Extra horizontal gap width between segments. If non-positive, the script uses `box_size`.

### Fonts and labels

* `--font-size-multiplier F` (`1.2`)
  Base font size is `box_size * font_size_multiplier`. Used for most labels by default.

* `--font-size-multiplier-stats F` (defaults to `font-size-multiplier` if not set)
  Multiplier used for the stats block font size.

* `--font-size-multiplier-notes F` (defaults to `font-size-multiplier` if not set)
  Multiplier used for the events/goals notes block font size.

* `--font-path PATH`
  Optional path to a TTF/OTF font file. If not provided or if loading fails, the script tries several common fonts (`Cantarell`, `DejaVuSerif`, `DejaVuSans`, `Times New Roman`) and then falls back to Pillow’s default font.

* `--show-start-end-labels`
  Show labels at the beginning and end of the life grid (right side).

* `--start-label TEXT` (`Start`)
  Text label for the first year row.

* `--end-label TEXT` (`Fin.`)
  Text label for the last year row.

### Stats and current-year highlighting

* `--show-stats`
  Draw a stats block summarizing:

  * Today’s date and age in years (as a float).
  * Weeks lived and weeks left in the configured grid.
  * Weeks used and remaining in the current life year.

  The script tries to place this block in the strip to the left of the grid. If there is not enough space, it falls back to a centered placement near the top of the image.

* `--highlight-current-year`
  Highlight the current year row with a background color.

* `--current-year-background-color HEX` (`#FFF4CC`)
  Background color for the highlighted current year row.

### Life expectancy and “bonus years”

* `--expected-years N`
  Expected lifespan in years. If set and `0 < N < --years`, the script draws a horizontal marker line between rows `N-1` and `N`. This corresponds to the transition from age `N-1` to age `N`.

* `--expectation-line-color HEX` (`#000000`)
  Color of the life expectancy line.

* `--shade-bonus-years`
  When used together with `--expected-years`, rows beyond the expected age are shaded using `--bonus-background-color`.

* `--bonus-background-color HEX` (`#F5F5F5`)
  Background color for shaded “bonus” years. If the current year row is highlighted and also lies in the bonus region, the highlight color takes precedence on that row.

### Events and goals

* `--event YYYY-MM-DD[:Description]`
  Define a single event from the command line.

* `--event-file PATH`
  Add events from a file (format described above). You can pass this multiple times.

* `--goal START_DATE:END_DATE[:Description]`
  Define a single goal period from the command line.

* `--goal-file PATH`
  Add goals from a file (format described above). You can pass this multiple times.

### Wallpaper integration

* `--set-wallpaper CMD_TEMPLATE`
  Shell command to set the wallpaper. The substring `{image}` is replaced with the absolute path of the generated image. The command is executed via `subprocess.run(..., shell=True, check=False)`; non-zero exit status does not raise an exception, but failures to start the command are logged as warnings.

Example for `nitrogen`:

```bash
--set-wallpaper "nitrogen --set-zoom-fill {image} --save"
```

---

## Example: Nitrogen Integration on Arch

A typical “daily wallpaper” configuration might look like this:

```bash
python ~/.config/memento-mori/memento-mori.py \
  --birthdate 2000-01-01 \
  --image-size 1920x1080 \
  --spacing 2 \
  --margin 40 \
  --box-size 9 \
  --theme warm-dark \
  --show-stats \
  --highlight-current-year \
  --show-start-end-labels \
  --expected-years 60 \
  --expectation-line-color "#FFFFFF" \
  --font-size-multiplier 1.6 \
  --goal-file ~/goals.txt \
  --event-file ~/events.txt \
  --font-path /usr/share/fonts/cantarell/Cantarell-VF.otf \
  --output ~/Pictures/memento_mori.png \
  --set-wallpaper "nitrogen --set-zoom-fill {image} --save"
```

Test this manually first. Once you are happy with the result, automate it as described below.

---

## Automation with systemd (user services)

On a modern Arch system, user-level `systemd` units are a robust way to schedule and trigger the generator.

### User service

Create the service unit:

```bash
mkdir -p ~/.config/systemd/user
nano ~/.config/systemd/user/memento-mori.service
```

Example `~/.config/systemd/user/memento-mori.service`:

```ini
[Unit]
Description=Generate memento mori wallpaper

[Service]
Type=oneshot
Environment=DISPLAY=:0
Environment=XAUTHORITY=%h/.Xauthority
ExecStart=/usr/bin/python %h/.config/memento-mori/memento-mori.py --birthdate 2000-01-01 --image-size 1920x1080 --spacing 2 --margin 40 --box-size 9 --theme warm-dark --show-stats --highlight-current-year --show-start-end-labels --expected-years 60 --expectation-line-color "#FFFFFF" --font-size-multiplier 1.6 --goal-file %h/goals.txt --event-file %h/events.txt --font-path /usr/share/fonts/cantarell/Cantarell-VF.otf --output %h/Pictures/memento_mori.png --set-wallpaper "nitrogen --set-zoom-fill %h/Pictures/memento_mori.png --save"
```

If you want to watch different paths, you might want to add a `--force` version of the above:

Example `~/.config/systemd/user/memento-mori-force.service`:

```ini
[Unit]
Description=Generate memento mori wallpaper

[Service]
Type=oneshot
Environment=DISPLAY=:0
Environment=XAUTHORITY=%h/.Xauthority
ExecStart=/usr/bin/python %h/.config/memento-mori/memento-mori.py --force --birthdate 2000-01-01 --image-size 1920x1080 --spacing 2 --margin 40 --box-size 9 --theme warm-dark --show-stats --highlight-current-year --show-start-end-labels --expected-years 60 --expectation-line-color "#FFFFFF" --font-size-multiplier 1.6 --goal-file %h/goals.txt --event-file %h/events.txt --font-path /usr/share/fonts/cantarell/Cantarell-VF.otf --output %h/Pictures/memento_mori.png --set-wallpaper "nitrogen --set-zoom-fill %h/Pictures/memento_mori.png --save"
```

Reload user units:

```bash
systemctl --user daemon-reload
```

From here, you can attach timers or path units to this service.

### Daily at 09:00

Create `~/.config/systemd/user/memento-mori-daily.timer`:

```ini
[Unit]
Description=Daily memento mori wallpaper update at 09:00

[Timer]
OnCalendar=*-*-* 09:00
Persistent=true
Unit=memento-mori.service

[Install]
WantedBy=timers.target
```

Activate:

```bash
systemctl --user enable --now memento-mori-daily.timer
```

### At login (shortly after boot)

Create `~/.config/systemd/user/memento-mori-login.timer`:

```ini
[Unit]
Description=Run memento mori wallpaper generator shortly after login

[Timer]
OnBootSec=1min
Unit=memento-mori.service

[Install]
WantedBy=timers.target
```

Activate:

```bash
systemctl --user enable --now memento-mori-login.timer
```

### When goals/events files change

Create a path unit `~/.config/systemd/user/memento-mori-goals.path`:

```ini
[Unit]
Description=Watch goals and events files for changes

[Path]
PathChanged=%h/goals.txt
PathChanged=%h/events.txt
Unit=memento-mori-force.service

[Install]
WantedBy=default.target
```

Activate:

```bash
systemctl --user enable --now memento-mori-goals.path
```

With the default caching behavior, changing `goals.txt` or `events.txt` will trigger the service, but the image will only be regenerated when the week changes. Add `--force` to the service’s `ExecStart` if you want each edit to force a redraw.

### Approximate “on wake/unlock”

A simple approximation is to run the service periodically while your user session is active and rely on the cache to limit actual work. Create `~/.config/systemd/user/memento-mori-periodic.timer`:

```ini
[Unit]
Description=Periodically refresh memento mori wallpaper

[Timer]
OnUnitInactiveSec=30min
Unit=memento-mori.service

[Install]
WantedBy=timers.target
```

Activate:

```bash
systemctl --user enable --now memento-mori-periodic.timer
```

Because of the cache, this typically does nothing except when a new week starts or when `--force` is enabled.

---

## Cron alternative (daily at 09:00)

If you prefer cron, install and enable `cronie`, then edit your user crontab:

```bash
crontab -e
```

Example entry:

```cron
0 9 * * * /usr/bin/python /home/youruser/.config/memento-mori/memento-mori.py \
  --birthdate 2000-01-01 \
  --image-size 1920x1080 \
  --spacing 2 \
  --margin 40 \
  --box-size 9 \
  --theme warm-dark \
  --show-stats \
  --highlight-current-year \
  --show-start-end-labels \
  --expected-years 60 \
  --expectation-line-color "#FFFFFF" \
  --font-size-multiplier 1.6 \
  --goal-file /home/youruser/goals.txt \
  --event-file /home/youruser/events.txt \
  --font-path /usr/share/fonts/cantarell/Cantarell-VF.otf \
  --output /home/youruser/Pictures/memento_mori.png \
  --set-wallpaper "nitrogen --set-zoom-fill /home/youruser/Pictures/memento_mori.png --save"
```

Replace `youruser` with your actual username.

Cron does not have native “on file change” or “on unlock” triggers; those are better handled by `systemd.path` units and timers.

---

## Notes and Troubleshooting

* If you see `This script requires Pillow. Install with: pip install pillow`, make sure you installed `pillow` into the environment from which you are running the script.
* If the script warns about missing fonts, either install the referenced fonts or provide a working `--font-path`.
* If you specify `--image-size` and get an error about the grid not fitting, you need to:

  * Increase `--image-size`, or
  * Decrease `--box-size`, `--spacing`, or `--years`, or
  * Disable or reduce grouping and extra gaps.
* The `--output` path should include a directory component. The script calls `os.makedirs(os.path.dirname(output_path), exist_ok=True)` before saving, so a bare filename (without a directory) is not currently supported.
* When setting up automation, always test the full `ExecStart` command manually in a terminal before relying on timers or cron.

